/*
 * Copyright (c) 1999, 2021, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package com.sun.tools.javac.jvm;

import java.io.*;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.CharBuffer;
import java.nio.file.ClosedFileSystemException;
import java.util.Arrays;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.function.IntFunction;

import javax.lang.model.element.Modifier;
import javax.lang.model.element.NestingKind;
import javax.tools.JavaFileManager;
import javax.tools.JavaFileObject;

import com.sun.tools.javac.code.Source.Feature;
import com.sun.tools.javac.comp.Annotate;
import com.sun.tools.javac.comp.Annotate.AnnotationTypeCompleter;
import com.sun.tools.javac.code.*;
import com.sun.tools.javac.code.Directive.*;
import com.sun.tools.javac.code.Lint.LintCategory;
import com.sun.tools.javac.code.Scope.WriteableScope;
import com.sun.tools.javac.code.Symbol.*;
import com.sun.tools.javac.code.Symtab;
import com.sun.tools.javac.code.Type.*;
import com.sun.tools.javac.comp.Annotate.AnnotationTypeMetadata;
import com.sun.tools.javac.file.BaseFileManager;
import com.sun.tools.javac.file.PathFileObject;
import com.sun.tools.javac.jvm.ClassFile.Version;
import com.sun.tools.javac.jvm.PoolConstant.NameAndType;
import com.sun.tools.javac.main.Option;
import com.sun.tools.javac.resources.CompilerProperties.Fragments;
import com.sun.tools.javac.resources.CompilerProperties.Warnings;
import com.sun.tools.javac.util.*;
import com.sun.tools.javac.util.DefinedBy.Api;
import com.sun.tools.javac.util.JCDiagnostic.DiagnosticPosition;

import static com.sun.tools.javac.code.Flags.*;
import static com.sun.tools.javac.code.Kinds.Kind.*;

import com.sun.tools.javac.code.Scope.LookupKind;

import static com.sun.tools.javac.code.TypeTag.ARRAY;
import static com.sun.tools.javac.code.TypeTag.CLASS;
import static com.sun.tools.javac.code.TypeTag.TYPEVAR;
import static com.sun.tools.javac.jvm.ClassFile.*;
import static com.sun.tools.javac.jvm.ClassFile.Version.*;

import static com.sun.tools.javac.main.Option.PARAMETERS;

/** This class provides operations to read a classfile into an internal
 *  representation. The internal representation is anchored in a
 *  ClassSymbol which contains in its scope symbol representations
 *  for all other definitions in the classfile. Top-level Classes themselves
 *  appear as members of the scopes of PackageSymbols.
 *
 *  <p><b>This is NOT part of any supported API.
 *  If you write code that depends on this, you do so at your own risk.
 *  This code and its internal interfaces are subject to change or
 *  deletion without notice.</b>
 */
public class ClassReader {
    /** The context key for the class reader. */
    protected static final Context.Key<ClassReader> classReaderKey = new Context.Key<>();

    public static final int INITIAL_BUFFER_SIZE = 0x0fff0;

    private final Annotate annotate;

    /** Switch: verbose output.
     */
    boolean verbose;

    /** Switch: allow modules.
     */
    boolean allowModules;

    /** Switch: allow sealed
     */
    boolean allowSealedTypes;

    /** Switch: allow records
     */
    boolean allowRecords;

   /** Lint option: warn about classfile issues
     */
    boolean lintClassfile;

    /** Switch: preserve parameter names from the variable table.
     */
    public boolean saveParameterNames;

    /**
     * The currently selected profile.
     */
    public final Profile profile;

    /** The log to use for verbose output
     */
    final Log log;

    /** The symbol table. */
    Symtab syms;

    Types types;

    /** The name table. */
    final Names names;

    /** Access to files
     */
    private final JavaFileManager fileManager;

    /** Factory for diagnostics
     */
    JCDiagnostic.Factory diagFactory;

    DeferredCompletionFailureHandler dcfh;

    /**
     * Support for preview language features.
     */
    Preview preview;

    /** The current scope where type variables are entered.
     */
    protected WriteableScope typevars;

    private List<InterimUsesDirective> interimUses = List.nil();
    private List<InterimProvidesDirective> interimProvides = List.nil();

    /** The path name of the class file currently being read.
     */
    protected JavaFileObject currentClassFile = null;

    /** The class or method currently being read.
     */
    protected Symbol currentOwner = null;

    /** The module containing the class currently being read.
     */
    protected ModuleSymbol currentModule = null;

    /** The buffer containing the currently read class file.
     */
    ByteBuffer buf = new ByteBuffer(INITIAL_BUFFER_SIZE);

    /** The current input pointer.
     */
    protected int bp;

    /** The pool reader.
     */
    PoolReader poolReader;

    /** The major version number of the class file being read. */
    int majorVersion;
    /** The minor version number of the class file being read. */
    int minorVersion;

    /** A table to hold the constant pool indices for method parameter
     * names, as given in LocalVariableTable attributes.
     */
    int[] parameterNameIndices;

    /**
     * A table to hold the access flags of the method parameters.
     */
    int[] parameterAccessFlags;

    /**
     * A table to hold annotations for method parameters.
     */
    ParameterAnnotations[] parameterAnnotations;

    /**
     * A holder for parameter annotations.
     */
    static class ParameterAnnotations {
        List<CompoundAnnotationProxy> proxies;

        void add(List<CompoundAnnotationProxy> newAnnotations) {
            if (proxies == null) {
                proxies = newAnnotations;
            } else {
                proxies = proxies.prependList(newAnnotations);
            }
        }
    }

    /**
     * Whether or not any parameter names have been found.
     */
    boolean haveParameterNameIndices;

    /** Set this to false every time we start reading a method
     * and are saving parameter names.  Set it to true when we see
     * MethodParameters, if it's set when we see a LocalVariableTable,
     * then we ignore the parameter names from the LVT.
     */
    boolean sawMethodParameters;

    /**
     * The set of attribute names for which warnings have been generated for the current class
     */
    Set<Name> warnedAttrs = new HashSet<>();

    /**
     * The prototype @Target Attribute.Compound if this class is an annotation annotated with
     * @Target
     */
    CompoundAnnotationProxy target;

    /**
     * The prototype @Repeatable Attribute.Compound if this class is an annotation annotated with
     * @Repeatable
     */
    CompoundAnnotationProxy repeatable;

    /** Get the ClassReader instance for this invocation. */
    public static ClassReader instance(Context context) {
        ClassReader instance = context.get(classReaderKey);
        if (instance == null)
            instance = new ClassReader(context);
        return instance;
    }

    /** Construct a new class reader. */
    protected ClassReader(Context context) {
        context.put(classReaderKey, this);
        annotate = Annotate.instance(context);
        names = Names.instance(context);
        syms = Symtab.instance(context);
        types = Types.instance(context);
        fileManager = context.get(JavaFileManager.class);
        if (fileManager == null)
            throw new AssertionError("FileManager initialization error");
        diagFactory = JCDiagnostic.Factory.instance(context);
        dcfh = DeferredCompletionFailureHandler.instance(context);

        log = Log.instance(context);

        Options options = Options.instance(context);
        verbose         = options.isSet(Option.VERBOSE);

        Source source = Source.instance(context);
        preview = Preview.instance(context);
        allowModules     = Feature.MODULES.allowedInSource(source);
        allowRecords = Feature.RECORDS.allowedInSource(source);
        allowSealedTypes = Feature.SEALED_CLASSES.allowedInSource(source);

        saveParameterNames = options.isSet(PARAMETERS);

        profile = Profile.instance(context);

        typevars = WriteableScope.create(syms.noSymbol);

        lintClassfile = Lint.instance(context).isEnabled(LintCategory.CLASSFILE);

        initAttributeReaders();
    }

    /** Add member to class unless it is synthetic.
     */
    private void enterMember(ClassSymbol c, Symbol sym) {
        // Synthetic members are not entered -- reason lost to history (optimization?).
        // Lambda methods must be entered because they may have inner classes (which reference them)
        if ((sym.flags_field & (SYNTHETIC|BRIDGE)) != SYNTHETIC || sym.name.startsWith(names.lambda))
            c.members_field.enter(sym);
    }

/************************************************************************
 * Error Diagnoses
 ***********************************************************************/

    public ClassFinder.BadClassFile badClassFile(String key, Object... args) {
        return new ClassFinder.BadClassFile (
            currentOwner.enclClass(),
            currentClassFile,
            diagFactory.fragment(key, args),
            diagFactory,
            dcfh);
    }

    public ClassFinder.BadEnclosingMethodAttr badEnclosingMethod(Symbol sym) {
        return new ClassFinder.BadEnclosingMethodAttr (
            currentOwner.enclClass(),
            currentClassFile,
            diagFactory.fragment(Fragments.BadEnclosingMethod(sym)),
            diagFactory,
            dcfh);
    }

/************************************************************************
 * Buffer Access
 ***********************************************************************/

    /** Read a character.
     */
    char nextChar() {
        char res = buf.getChar(bp);
        bp += 2;
        return res;
    }

    /** Read a byte.
     */
    int nextByte() {
        return buf.getByte(bp++) & 0xFF;
    }

    /** Read an integer.
     */
    int nextInt() {
        int res = buf.getInt(bp);
        bp += 4;
        return res;
    }

/************************************************************************
 * Constant Pool Access
 ***********************************************************************/

    /** Read module_flags.
     */
    Set<ModuleFlags> readModuleFlags(int flags) {
        Set<ModuleFlags> set = EnumSet.noneOf(ModuleFlags.class);
        for (ModuleFlags f : ModuleFlags.values()) {
            if ((flags & f.value) != 0)
                set.add(f);
        }
        return set;
    }

    /** Read resolution_flags.
     */
    Set<ModuleResolutionFlags> readModuleResolutionFlags(int flags) {
        Set<ModuleResolutionFlags> set = EnumSet.noneOf(ModuleResolutionFlags.class);
        for (ModuleResolutionFlags f : ModuleResolutionFlags.values()) {
            if ((flags & f.value) != 0)
                set.add(f);
        }
        return set;
    }

    /** Read exports_flags.
     */
    Set<ExportsFlag> readExportsFlags(int flags) {
        Set<ExportsFlag> set = EnumSet.noneOf(ExportsFlag.class);
        for (ExportsFlag f: ExportsFlag.values()) {
            if ((flags & f.value) != 0)
                set.add(f);
        }
        return set;
    }

    /** Read opens_flags.
     */
    Set<OpensFlag> readOpensFlags(int flags) {
        Set<OpensFlag> set = EnumSet.noneOf(OpensFlag.class);
        for (OpensFlag f: OpensFlag.values()) {
            if ((flags & f.value) != 0)
                set.add(f);
        }
        return set;
    }

    /** Read requires_flags.
     */
    Set<RequiresFlag> readRequiresFlags(int flags) {
        Set<RequiresFlag> set = EnumSet.noneOf(RequiresFlag.class);
        for (RequiresFlag f: RequiresFlag.values()) {
            if ((flags & f.value) != 0)
                set.add(f);
        }
        return set;
    }

/************************************************************************
 * Reading Types
 ***********************************************************************/

    /** The unread portion of the currently read type is
     *  signature[sigp..siglimit-1].
     */
    byte[] signature;
    int sigp;
    int siglimit;
    boolean sigEnterPhase = false;

    /** Convert signature to type, where signature is a byte array segment.
     */
    Type sigToType(byte[] sig, int offset, int len) {
        signature = sig;
        sigp = offset;
        siglimit = offset + len;
        return sigToType();
    }

    /** Convert signature to type, where signature is implicit.
     */
    Type sigToType() {
        switch ((char) signature[sigp]) {
        case 'T':
            sigp++;
            int start = sigp;
            while (signature[sigp] != ';') sigp++;
            sigp++;
            return sigEnterPhase
                ? Type.noType
                : findTypeVar(names.fromUtf(signature, start, sigp - 1 - start));
        case '+': {
            sigp++;
            Type t = sigToType();
            return new WildcardType(t, BoundKind.EXTENDS, syms.boundClass);
        }
        case '*':
            sigp++;
            return new WildcardType(syms.objectType, BoundKind.UNBOUND,
                                    syms.boundClass);
        case '-': {
            sigp++;
            Type t = sigToType();
            return new WildcardType(t, BoundKind.SUPER, syms.boundClass);
        }
        case 'B':
            sigp++;
            return syms.byteType;
        case 'C':
            sigp++;
            return syms.charType;
        case 'D':
            sigp++;
            return syms.doubleType;
        case 'F':
            sigp++;
            return syms.floatType;